<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Scan QR Code</title>
<style>
    body {
        margin: 0;
        font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
        background: #eef1f5;
        display: flex;
        justify-content: center;
        align-items: center;
        min-height: 100vh;
        padding: 20px;
    }
    .container {
        background: #ffffff;
        padding: 40px 45px;
        border-radius: 12px;
        width: 100%;
        max-width: 500px;
        box-shadow: 0 5px 25px rgba(0,0,0,0.12);
        text-align: center;
    }
    h3 {
        margin-bottom: 28px;
        color: #222;
        font-size: 24px;
        font-weight: 600;
    }
    #video {
        width: 100%;
        max-width: 300px;
        border: 1px solid #cfcfcf;
        border-radius: 6px;
        margin-bottom: 20px;
    }
    #result {
        width: 100%;
        max-width: 100%;
        margin-top: 20px;
        padding: 10px;
        background: #f1f1f1;
        border: 1px solid #cfcfcf;
        border-radius: 6px;
        font-size: 14px;
        color: #222;
        word-wrap: break-word;
        resize: vertical;
        min-height: 80px;
    }
    .error {
        color: #dc3545;
        font-size: 14px;
        margin-top: 10px;
    }
    .success {
        color: #28a745;
        font-size: 14px;
        margin-top: 10px;
    }
    
    /* Student Info Display */
    #studentInfo {
        margin-top: 20px;
        padding: 20px;
        background: #f8f9fa;
        border: 2px solid #28a745;
        border-radius: 8px;
        text-align: left;
        display: none;
    }
    #studentInfo.show {
        display: block;
    }
    #studentInfo h4 {
        margin-top: 0;
        color: #28a745;
        text-align: center;
    }
    #studentInfo p {
        margin: 8px 0;
        font-size: 15px;
        color: #333;
    }
    #studentInfo strong {
        color: #0a6cff;
    }
    #studentInfo img {
        display: block;
        margin: 15px auto 0;
        max-width: 150px;
        border: 2px solid #ddd;
        border-radius: 6px;
    }
    
    .loading {
        color: #0a6cff;
        font-style: italic;
    }
</style>
<script src="https://unpkg.com/@zxing/library@0.19.3/umd/index.min.js"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</head>
<body>
<div class="container">
    <h3>Scan QR Code</h3>
    <p>Allow camera access and point it at a QR code.</p>
    
    <video id="video" autoplay muted playsinline></video>
    
    <textarea id="result" readonly placeholder="Scanned QR content will appear here..."></textarea>
    
    <div id="message"></div>
    
    <!-- Student Information Display -->
    <div id="studentInfo">
        <h4>✓ Student Found</h4>
        <p><strong>Name:</strong> <span id="studentName"></span></p>
        <p><strong>Course:</strong> <span id="studentCourse"></span></p>
        <p><strong>Year Level:</strong> <span id="studentYear"></span></p>
        <img id="studentQR" src="" alt="QR Code">
    </div>
</div>

<script>
let codeReader;
const video = document.getElementById('video');
const resultTextarea = document.getElementById('result');
const messageDiv = document.getElementById('message');
const studentInfoDiv = document.getElementById('studentInfo');
let lastResult = null; // Track last scanned QR to avoid repeated searches

// Function to search student in database via AJAX
function searchStudent(qrText) {
    // Show loading message
    messageDiv.className = 'loading';
    messageDiv.textContent = 'Searching database...';
    studentInfoDiv.classList.remove('show');
    
    $.ajax({
        url: "<?= base_url('appcontroller/searchqrcode'); ?>",
        type: "POST",
        data: { qr_text: qrText },
        dataType: "json",
        
        success: function(res) {
            console.log('Response:', res);
            
            if (res.status === 'success') {
                // Student found - display info
                messageDiv.className = 'success';
                messageDiv.textContent = '✓ Student record found!';
                
                // Populate student info
                document.getElementById('studentName').textContent = res.data.Name;
                document.getElementById('studentCourse').textContent = res.data.Course;
                document.getElementById('studentYear').textContent = res.data.Year_level;
                document.getElementById('studentQR').src = '<?= base_url('assets/qrcode/'); ?>' + res.data.qr_image;
                
                // Show the info card
                studentInfoDiv.classList.add('show');
                
            } else {
                // Student not found
                messageDiv.className = 'error';
                messageDiv.textContent = '✗ ' + (res.message || 'Student not found in database');
                studentInfoDiv.classList.remove('show');
            }
        },
        
        error: function(xhr, status, error) {
            console.error('AJAX Error:', xhr.responseText);
            messageDiv.className = 'error';
            messageDiv.textContent = '✗ Error searching database: ' + error;
            studentInfoDiv.classList.remove('show');
        }
    });
}

// Start continuous QR scanning
async function startContinuousScan() {
    codeReader = new ZXing.BrowserQRCodeReader();
    
    try {
        const devices = await navigator.mediaDevices.enumerateDevices();
        const videoDevices = devices.filter(d => d.kind === 'videoinput');
        
        if (videoDevices.length === 0) {
            messageDiv.className = 'error';
            messageDiv.textContent = 'No cameras detected. Check permissions.';
            return;
        }
        
        const selectedDeviceId = videoDevices[0].deviceId;
        
        await codeReader.decodeFromVideoDevice(selectedDeviceId, video, (result, err) => {
            if (result) {
                // Only search if the QR code is new
                if (result.text !== lastResult) {
                    lastResult = result.text;
                    resultTextarea.value = result.text;
                    
                    // Automatically search in database
                    searchStudent(result.text);
                }
            }
            
            if (err && !(err instanceof ZXing.NotFoundException)) {
                console.error('Scan error:', err);
                messageDiv.className = 'error';
                messageDiv.textContent = 'Scan error: ' + err.message;
            }
        });
        
        resultTextarea.value = 'Scanning... Point at QR code.';
        messageDiv.textContent = '';
        
    } catch (error) {
        console.error('Error starting scanner:', error);
        messageDiv.className = 'error';
        messageDiv.textContent = 'Camera access denied: ' + error.message;
    }
}

// Start scanning when page loads
window.addEventListener('load', startContinuousScan);
</script>
</body>
</html>