<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Registration extends CI_Controller {
    
    public function __construct() {
        parent::__construct();
        $this->load->model('Student_model');
        $this->load->library('form_validation');
        $this->load->library('session');
        $this->load->helper('url');
        $this->load->config('email');
        $this->load->database();
    }
    
    public function index() {
        $this->load->view('registration/form');
    }
    
    public function save() {
        // Form validation rules
        $this->form_validation->set_rules('studentno', 'Student Number', 'required|trim');
        $this->form_validation->set_rules('lastname', 'Last Name', 'required|trim');
        $this->form_validation->set_rules('firstname', 'First Name', 'required|trim');
        $this->form_validation->set_rules('course', 'Course', 'required');
        $this->form_validation->set_rules('emailaddress', 'Email Address', 'required|valid_email|trim');
        $this->form_validation->set_rules('contactno', 'Contact Number', 'required|regex_match[/^[0-9]{10,11}$/]|trim');
        
        if ($this->form_validation->run() == FALSE) {
            $this->session->set_flashdata('error', validation_errors());
            redirect('registration');
            return;
        }
        
        // Check duplicates
        if ($this->Student_model->check_email_exists($this->input->post('emailaddress'))) {
            $this->session->set_flashdata('error', 'Email address already registered!');
            redirect('registration');
            return;
        }
        
        if ($this->Student_model->check_studentno_exists($this->input->post('studentno'))) {
            $this->session->set_flashdata('error', 'Student number already registered!');
            redirect('registration');
            return;
        }
        
        // Prepare data
        $data = array(
            'studentno' => $this->input->post('studentno'),
            'lastname' => $this->input->post('lastname'),
            'firstname' => $this->input->post('firstname'),
            'course' => $this->input->post('course'),
            'emailaddress' => $this->input->post('emailaddress'),
            'contactno' => $this->input->post('contactno')
        );
        
        // Insert student
        $result = $this->Student_model->insert_student($data);
        
        if ($result['status']) {
            // Send verification email
            if ($this->send_verification_email($result['email'], $result['token'], $data['firstname'])) {
                $this->session->set_flashdata('success', 'Registration successful! Please check your email for verification link.');
            } else {
                $this->session->set_flashdata('error', 'Registration successful but email sending failed. Please contact support.');
            }
        } else {
            $this->session->set_flashdata('error', 'Registration failed. Please try again.');
        }
        
        redirect('registration');
    }
    
    private function send_verification_email($email, $token, $firstname) {
        // Load PHPMailer
        require_once APPPATH . 'third_party/PHPMailer/PHPMailer.php';
        require_once APPPATH . 'third_party/PHPMailer/SMTP.php';
        require_once APPPATH . 'third_party/PHPMailer/Exception.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            // Server settings
            $mail->isSMTP();
            $mail->Host = $this->config->item('smtp_host');
            $mail->SMTPAuth = true;
            $mail->Username = $this->config->item('smtp_user');
            $mail->Password = $this->config->item('smtp_pass');
            $mail->SMTPSecure = $this->config->item('smtp_crypto');
            $mail->Port = $this->config->item('smtp_port');
            
            // Recipients
            $mail->setFrom($this->config->item('from_email'), $this->config->item('from_name'));
            $mail->addAddress($email);
            
            // Content
            $mail->isHTML(true);
            $mail->Subject = 'Student Registration - Email Verification';
            
            $verification_link = base_url('registration/verify?token=' . $token);
            
            // Load email template
            $email_data = array(
                'firstname' => $firstname,
                'verification_link' => $verification_link
            );
            
            $mail->Body = $this->load->view('templates/email_verification', $email_data, TRUE);
            
            $mail->send();
            return TRUE;
        } catch (Exception $e) {
            log_message('error', 'Email sending failed: ' . $mail->ErrorInfo);
            return FALSE;
        }
    }
    
    public function verify() {
        $token = $this->input->get('token');
        
        if ($token) {
            if ($this->Student_model->verify_email($token)) {
                $data['message'] = 'Email verified successfully! You can now use the system.';
                $data['type'] = 'success';
            } else {
                $data['message'] = 'Invalid or expired verification link.';
                $data['type'] = 'error';
            }
        } else {
            $data['message'] = 'No verification token provided.';
            $data['type'] = 'error';
        }
        
        $this->load->view('registration/verify', $data);
    }
}